/*
 * Decompiled with CFR 0.152.
 */
package dev.felnull.imp.client.lava;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import dev.felnull.imp.IamMusicPlayer;
import dev.felnull.imp.client.lava.LavaPlayerLoader;
import dev.felnull.imp.include.dev.felnull.fnjl.util.FNDataUtil;
import dev.felnull.imp.include.org.apache.commons.codec.binary.Hex;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.io.UncheckedIOException;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class LavaNativeManager {
    private static final Logger LOGGER = LogManager.getLogger(LavaNativeManager.class);
    private static final Gson GSON = new Gson();
    private static final LavaNativeManager INSTANCE = new LavaNativeManager();
    private static final String NATIVES_VERSION = "2.2.4";
    private static final String HASH_FILE_NAME = "hash.json";
    private static final int CONNECTION_TIMEOUT = 10000;
    private static final int READ_TIMEOUT = 30000;
    private static final int DOWNLOAD_RETRY_COUNT = 3;
    private static final long DOWNLOAD_RETRY_DELAY_MS = 1000L;
    private final ExecutorService downloadExecutor = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setNameFormat("lava-natives-downloader-%d").setDaemon(true).build());
    private final Map<String, CompletableFuture<Boolean>> activeDownloads = new ConcurrentHashMap<String, CompletableFuture<Boolean>>();

    private LavaNativeManager() {
    }

    public static LavaNativeManager getInstance() {
        return INSTANCE;
    }

    public boolean load(String osAndArch, String name) {
        Path nativesDir = LavaPlayerLoader.getNaiveLibraryFolder().resolve(osAndArch);
        File nativesDirFile = nativesDir.toFile();
        Path nativeLibPath = nativesDir.resolve(name);
        if (!this.isValidNativesDirectory(nativesDirFile)) {
            try {
                LOGGER.info("LavaPlayer natives for {} need to be downloaded", (Object)osAndArch);
                boolean success = this.downloadNatives(osAndArch).join();
                if (!success) {
                    LOGGER.error("LavaPlayer natives download failed for {}", (Object)osAndArch);
                    return false;
                }
                LOGGER.info("LavaPlayer natives download successful for {}", (Object)osAndArch);
            }
            catch (Exception e) {
                LOGGER.error("LavaPlayer natives download failed for {}", (Object)osAndArch, (Object)e);
                return false;
            }
        }
        LOGGER.info("LavaPlayer native({}) check successful", (Object)name);
        return Files.exists(nativeLibPath, new LinkOption[0]);
    }

    public CompletableFuture<Boolean> downloadNatives(String osAndArch) {
        return this.activeDownloads.computeIfAbsent(osAndArch, key -> CompletableFuture.supplyAsync(() -> {
            try {
                Boolean bl = this.downloadAndExtractNatives(osAndArch);
                return bl;
            }
            catch (Exception e) {
                LOGGER.error("Failed to download natives for {}", (Object)osAndArch, (Object)e);
                Boolean bl = false;
                return bl;
            }
            finally {
                this.activeDownloads.remove(osAndArch);
            }
        }, this.downloadExecutor));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean downloadAndExtractNatives(String osAndArch) throws Exception {
        Path nativesDir = LavaPlayerLoader.getNaiveLibraryFolder().resolve(osAndArch);
        File nativesDirFile = nativesDir.toFile();
        if (!nativesDirFile.exists() && !nativesDirFile.mkdirs()) {
            throw new IOException("Failed to create the directory for native libraries: " + String.valueOf(nativesDir));
        }
        JsonObject manifestJson = this.downloadManifest();
        if (!manifestJson.has(NATIVES_VERSION) || !manifestJson.get(NATIVES_VERSION).isJsonObject()) {
            throw new IllegalStateException("Native library version 2.2.4 not found in manifest. Available versions: " + String.valueOf(manifestJson.keySet()));
        }
        JsonObject versionJson = manifestJson.getAsJsonObject(NATIVES_VERSION);
        if (!versionJson.has(osAndArch) || !versionJson.get(osAndArch).isJsonObject()) {
            throw new IllegalStateException("Unsupported OS or architecture: " + osAndArch + ". Available platforms: " + String.valueOf(versionJson.keySet()));
        }
        JsonObject platformJson = versionJson.getAsJsonObject(osAndArch);
        if (!platformJson.has("hash")) {
            throw new IllegalStateException("Hash value not found for " + osAndArch);
        }
        if (!platformJson.has("url")) {
            throw new IllegalStateException("Download URL not found for " + osAndArch);
        }
        JsonObject hashJson = new JsonObject();
        hashJson.add("hash", platformJson.get("hash"));
        Files.writeString(nativesDir.resolve(HASH_FILE_NAME), (CharSequence)GSON.toJson((JsonElement)hashJson), new OpenOption[0]);
        String downloadUrl = platformJson.get("url").getAsString();
        LOGGER.info("Downloading natives from: {}", (Object)downloadUrl);
        Path tempFile = this.downloadWithRetry(new URI(downloadUrl).toURL(), nativesDir);
        try {
            this.extractNatives(tempFile, nativesDir);
        }
        finally {
            try {
                Files.deleteIfExists(tempFile);
            }
            catch (IOException e) {
                LOGGER.warn("Failed to delete temporary download file: {}", (Object)tempFile, (Object)e);
            }
        }
        if (!this.validateNativesIntegrity(nativesDirFile)) {
            try {
                this.deleteDirectoryContents(nativesDirFile);
            }
            catch (IOException e) {
                LOGGER.warn("Failed to clean up invalid natives directory: {}", (Object)nativesDir, (Object)e);
            }
            throw new IllegalStateException("Native library integrity check failed for " + osAndArch);
        }
        return true;
    }

    private JsonObject downloadManifest() throws Exception {
        String manifestUrlString = IamMusicPlayer.getConfig().lavaPlayerNativesURL;
        LOGGER.info("Downloading natives manifest from: {}", (Object)manifestUrlString);
        URL manifestUrl = new URI(manifestUrlString).toURL();
        HttpURLConnection connection = (HttpURLConnection)manifestUrl.openConnection();
        connection.setConnectTimeout(10000);
        connection.setReadTimeout(30000);
        connection.setRequestProperty("User-Agent", "IamMusicPlayer");
        int responseCode = connection.getResponseCode();
        if (responseCode != 200) {
            throw new IOException("Failed to download manifest. HTTP response code: " + responseCode);
        }
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));){
            JsonObject jsonObject = (JsonObject)GSON.fromJson((Reader)reader, JsonObject.class);
            return jsonObject;
        }
    }

    private Path downloadWithRetry(URL url, Path targetDir) throws Exception {
        Path tempFile = targetDir.resolve("dev.felnull.imp.include.natives_download_" + System.currentTimeMillis() + ".tmp");
        IOException lastException = null;
        for (int attempt = 1; attempt <= 3; ++attempt) {
            if (attempt > 1) {
                LOGGER.info("Retrying download (attempt {}/{})", (Object)attempt, (Object)3);
                Thread.sleep(1000L);
            }
            try {
                this.downloadFile(url, tempFile);
                LOGGER.info("Download completed successfully");
                return tempFile;
            }
            catch (IOException e) {
                lastException = e;
                LOGGER.warn("Download attempt {} failed: {}", (Object)attempt, (Object)e.getMessage());
                try {
                    Files.deleteIfExists(tempFile);
                }
                catch (IOException deleteEx) {
                    LOGGER.debug("Failed to delete partial download", (Throwable)deleteEx);
                }
                continue;
            }
        }
        throw new IOException("Failed to download after 3 attempts", lastException);
    }

    private void downloadFile(URL url, Path destination) throws IOException {
        HttpURLConnection connection = (HttpURLConnection)url.openConnection();
        connection.setConnectTimeout(10000);
        connection.setReadTimeout(30000);
        connection.setRequestProperty("User-Agent", "IamMusicPlayer");
        int responseCode = connection.getResponseCode();
        if (responseCode != 200) {
            throw new IOException("Failed to download file. HTTP response code: " + responseCode + " for URL: " + String.valueOf(url));
        }
        try (ReadableByteChannel readChannel = Channels.newChannel(connection.getInputStream());
             FileChannel writeChannel = FileChannel.open(destination, StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING);){
            long bytesTransferred;
            long fileSize = connection.getContentLengthLong();
            LOGGER.info("Downloading {} ({} bytes)", (Object)url, (Object)(fileSize > 0L ? String.format("%,d", fileSize) : "unknown size"));
            long position = 0L;
            long chunkSize = 0x100000L;
            while ((bytesTransferred = writeChannel.transferFrom(readChannel, position, chunkSize)) > 0L) {
                int progress;
                if (fileSize <= 0L || (progress = (int)((position += bytesTransferred) * 100L / fileSize)) % 10 != 0) continue;
                LOGGER.debug("Download progress: {}%", (Object)progress);
            }
            LOGGER.info("Downloaded {} bytes", (Object)position);
        }
    }

    private void extractNatives(Path zipFile, Path targetDir) throws IOException {
        LOGGER.info("Extracting natives from {} to {}", (Object)zipFile, (Object)targetDir);
        try (InputStream fileStream = Files.newInputStream(zipFile, new OpenOption[0]);
             BufferedInputStream bufferedStream = new BufferedInputStream(fileStream);
             ZipInputStream zipStream = new ZipInputStream(bufferedStream);){
            ZipEntry entry;
            int fileCount = 0;
            while ((entry = zipStream.getNextEntry()) != null) {
                String entryName;
                if (entry.isDirectory() || (entryName = entry.getName()).contains("__MACOSX") || entryName.startsWith("._")) continue;
                Path entryPath = targetDir.resolve(entryName);
                LOGGER.debug("Extracting: {}", (Object)entryName);
                Files.createDirectories(entryPath.getParent(), new FileAttribute[0]);
                try (OutputStream os = Files.newOutputStream(entryPath, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);){
                    int bytesRead;
                    byte[] buffer = new byte[8192];
                    while ((bytesRead = zipStream.read(buffer)) != -1) {
                        os.write(buffer, 0, bytesRead);
                    }
                    ++fileCount;
                }
                String fileName = entryPath.getFileName().toString();
                if (this.isUnixSystem() && (fileName.endsWith(".so") || fileName.endsWith(".dylib"))) {
                    entryPath.toFile().setExecutable(true, false);
                    LOGGER.debug("Set executable flag for: {}", (Object)fileName);
                }
                zipStream.closeEntry();
            }
            LOGGER.info("Extracted {} files", (Object)fileCount);
            if (fileCount == 0) {
                throw new IOException("No files were extracted from the archive");
            }
        }
    }

    private boolean isValidNativesDirectory(File directory) {
        if (!directory.exists() || !directory.isDirectory()) {
            return false;
        }
        return this.validateNativesIntegrity(directory);
    }

    private boolean validateNativesIntegrity(File directory) {
        try {
            JsonObject hashJson;
            LOGGER.debug("Validating native libraries integrity in {}", (Object)directory);
            File[] files = directory.listFiles();
            if (files == null) {
                LOGGER.error("Directory does not exist or cannot be read: {}", (Object)directory);
                return false;
            }
            List<File> relevantFiles = Arrays.stream(files).filter(f -> !f.isHidden()).filter(f -> !f.getName().equalsIgnoreCase("Thumbs.db")).filter(f -> !f.getName().startsWith(".")).collect(Collectors.toList());
            Optional<File> hashFile = relevantFiles.stream().filter(f -> f.getName().equals(HASH_FILE_NAME)).findAny();
            if (hashFile.isEmpty()) {
                LOGGER.error("Missing {} in directory: {}", (Object)HASH_FILE_NAME, (Object)directory);
                return false;
            }
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(hashFile.get())));){
                hashJson = (JsonObject)GSON.fromJson((Reader)reader, JsonObject.class);
            }
            if (!hashJson.has("hash")) {
                LOGGER.error("{} missing required field 'hash'", (Object)HASH_FILE_NAME);
                return false;
            }
            relevantFiles.remove(hashFile.get());
            List<File> nativeLibs = this.filterNativeLibrariesForCurrentOS(relevantFiles);
            if (nativeLibs.isEmpty()) {
                LOGGER.error("No native library files found in {}", (Object)directory);
                return false;
            }
            LOGGER.debug("Found {} native library file(s) to validate", (Object)nativeLibs.size());
            JsonElement hashElement = hashJson.get("hash");
            return this.validateHashes(hashElement, nativeLibs);
        }
        catch (Exception e) {
            LOGGER.error("Unexpected error during native libraries integrity check: {}", (Object)e.getMessage(), (Object)e);
            return false;
        }
    }

    private boolean validateHashes(JsonElement hashElement, List<File> nativeLibs) {
        if (hashElement.isJsonPrimitive()) {
            String actualHash;
            if (nativeLibs.isEmpty()) {
                LOGGER.error("No native library files found to validate against hash");
                return false;
            }
            if (nativeLibs.size() > 1) {
                LOGGER.warn("Primitive hash provided but {} files found (expected 1). Validating first file only.", (Object)nativeLibs.size());
            }
            File targetFile = nativeLibs.get(0);
            String expectedHash = hashElement.getAsString();
            if (!expectedHash.equals(actualHash = this.calculateMD5Hash(targetFile.toPath()))) {
                LOGGER.error("Hash mismatch for {}: expected {}, got {}", (Object)targetFile.getName(), (Object)expectedHash, (Object)actualHash);
                return false;
            }
            LOGGER.debug("Hash validated successfully for {}", (Object)targetFile.getName());
            return true;
        }
        if (hashElement.isJsonObject()) {
            HashSet<String> actualFiles;
            JsonObject hashesObject = hashElement.getAsJsonObject();
            Map<String, File> filesByName = nativeLibs.stream().collect(Collectors.toMap(File::getName, f -> f));
            for (Map.Entry entry : hashesObject.entrySet()) {
                String filename = (String)entry.getKey();
                String expectedHash = ((JsonElement)entry.getValue()).getAsString();
                File file = filesByName.get(filename);
                if (file == null) {
                    LOGGER.error("Expected file {} not found", (Object)filename);
                    return false;
                }
                String actualHash = this.calculateMD5Hash(file.toPath());
                if (!expectedHash.equals(actualHash)) {
                    LOGGER.error("Hash mismatch for {}: expected {}, got {}", (Object)filename, (Object)expectedHash, (Object)actualHash);
                    return false;
                }
                LOGGER.debug("Hash validated successfully for {}", (Object)filename);
            }
            HashSet expectedFiles = new HashSet(hashesObject.keySet());
            if (!expectedFiles.equals(actualFiles = new HashSet<String>(filesByName.keySet()))) {
                HashSet<String> extraFiles = new HashSet<String>(actualFiles);
                extraFiles.removeAll(expectedFiles);
                if (!extraFiles.isEmpty()) {
                    LOGGER.warn("Found extra files not in hash.json: {}", extraFiles);
                }
                HashSet missingFiles = new HashSet(expectedFiles);
                missingFiles.removeAll(actualFiles);
                if (!missingFiles.isEmpty()) {
                    LOGGER.error("Missing files that should be present according to hash.json: {}", missingFiles);
                    return false;
                }
            }
            LOGGER.debug("All hashes validated successfully");
            return true;
        }
        LOGGER.error("Invalid 'hash' format in {} (expected string or object). Found: {}", (Object)HASH_FILE_NAME, (Object)hashElement);
        return false;
    }

    private List<File> filterNativeLibrariesForCurrentOS(List<File> files) {
        String os = System.getProperty("os.name").toLowerCase();
        return files.stream().filter(f -> {
            String name = f.getName();
            if (os.contains("linux") && name.endsWith(".so")) {
                return true;
            }
            if (os.contains("mac") && name.endsWith(".dylib")) {
                return true;
            }
            return os.contains("win") && name.endsWith(".dll");
        }).collect(Collectors.toList());
    }

    private String calculateMD5Hash(Path path) {
        try {
            byte[] hash = FNDataUtil.createMD5Hash(Files.readAllBytes(path));
            return new String(Hex.encodeHex(hash));
        }
        catch (IOException | NoSuchAlgorithmException e) {
            throw new UncheckedIOException("Failed to calculate MD5 hash for " + String.valueOf(path), new IOException(e));
        }
    }

    private void deleteDirectoryContents(File directory) throws IOException {
        if (!directory.exists() || !directory.isDirectory()) {
            return;
        }
        File[] files = directory.listFiles();
        if (files == null) {
            return;
        }
        for (File file : files) {
            if (file.isDirectory()) {
                this.deleteDirectoryContents(file);
                if (file.delete()) continue;
                LOGGER.warn("Failed to delete directory: {}", (Object)file);
                continue;
            }
            if (file.delete()) continue;
            LOGGER.warn("Failed to delete file: {}", (Object)file);
        }
    }

    private boolean isUnixSystem() {
        String os = System.getProperty("os.name").toLowerCase();
        return os.contains("nix") || os.contains("nux") || os.contains("mac");
    }

    public void shutdown() {
        this.downloadExecutor.shutdown();
        try {
            if (!this.downloadExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.downloadExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.downloadExecutor.shutdownNow();
        }
    }
}

